#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
This module provides a class for the GUI that controls the communication
to the instruments for the CASAC experiment.

Note that this module is typically called directly as a standalone program.
When this is done, there are a number of optional commandline arguments
that can be used to enhanced functionality. If this is of interest to you,
try calling the file directly with the `-h` or `--help` argument.

Also, see the hover text that appears above each tab, for both generalized
operation, as well as keyboard shortcuts.

Copyright (c) 2020 pyLabSpec Development Team
Distributed under the 3-clause BSD license. See LICENSE for more infomation.
"""
# standard library
import sys
import os
import time
import datetime
import subprocess
import tempfile
import codecs
import re
from functools import partial
import math
import argparse
import struct
import distutils.version
# third-party
from scipy import fftpack, signal

import numpy as np
from PyQt5 import QtGui, QtCore
from PyQt5 import uic

try:
	import pyqtgraph as pg
	from pyqtgraph import siScale
except ImportError as e:
	msg = "Could not import pyqtgraph!"
	if 'anaconda' in str(sys.version).lower():
		msg += "\n\tTry doing something like 'conda install pyqtgraph'\n"
	elif sys.platform == 'darwin':
		msg += "\n\tTry doing something like 'sudo port install py-pyqtgraph'\n"
	else:
		msg += "\n\tTry doing something like 'sudo pip install pyqtgraph'\n"
	raise ImportError(msg)

from pyqtgraph.Qt import QtGui, QtCore
from pyqtgraph.Qt import uic
loadUiType = uic.loadUiType
# local
if not os.path.dirname(os.path.dirname(os.path.realpath(__file__))) in sys.path:
	sys.path.append(os.path.dirname(os.path.dirname(os.path.realpath(__file__))))
if not os.path.dirname(os.path.realpath(__file__)) in sys.path:
	sys.path.append(os.path.dirname(os.path.realpath(__file__)))
import Instruments
import Dialogs
from Dialogs import *
import Widgets
from Widgets import *
import DateAxisItem
from miscfunctions import *
from Instruments.instrument import Settings
from Instruments import scope as scopelib
from Instruments import awg as awglib
from Instruments import synthesizer
from Instruments import digitizer

import Spectrum.spectrum as spec
from settings import *

if sys.version_info[0] == 3:
    msg = "the chirp GUI has not been created/updated for python 3 compatibility!"
    msg += "\nthe following knwon issues exist:"
    msg += "\n\t- TabError: inconsistent use of tabs and spaces in indentation"
    raise UserWarning(msg)
if distutils.version.LooseVersion(pg.Qt.QtVersion) < "5":
    raise UserWarning("the chirp GUI has not been created/updated for PyQt4 compatibility!")


DIGI_TIMEOUT = 100000 # 100 sec
#probe = {}
#probe_avg = {}

class TestThread(QtCore.QThread):
    result = QtCore.pyqtSignal(object, object)

    def __init__(self,
                 parent = None):
        QtCore.QThread.__init__(self)

    def run(self):
        print("Wait for 5 seconds")
        self.msleep(5000)
        print("Wait Done")

class AcqThread(QtCore.QThread):

    result = QtCore.pyqtSignal(object, object)
    sigStatus = QtCore.pyqtSignal(str)
    sigScanNumber = QtCore.pyqtSignal(int)
    sigUpdatePlots = QtCore.pyqtSignal()

    def __init__(self,
                 parent = None):
        QtCore.QThread.__init__(self)
        self.current_run = 0
        self.isRunning = False

    def acquireData(self):
        # Start and Stop AWG before acquisition in order to be
        # sure that segments can be assigned correctly
        self.awg.start_immediate()
        self.awg.query('*OPC?')
        # Start single acquisition
        self.scope.write(':SINGLE')
        # wait until scope-trigger is armed 
        if int(self.scope.query(':AER?')) == 0:
            self.msleep(100)
        # fire awg trigger
        #self.awg.write(':TRIG:BEG')
        self.awg.write(':TRIG:ENAB')
        # wait until acquisition is done
        while int(self.scope.query(':ADER?')) == 0:
            self.msleep(100)
        # switch of awg
        self.awg.stop_immediate()

    def acquireDataDigi(self):
        """
        Start the data acquisition.
        """
        # Start and Stop AWG before acquisition in order to be sure that
        # segments can be assigned correctly
        dummy = self.awg.start_immediate()
        dummy = self.awg.query('*OPC?')
        status = self.scope.initiateAcquisition()
        time.sleep(1)
        self.awg.write(':TRIG:ENAB')
        self.scope.waitForAcquisitionComplete(timeout = DIGI_TIMEOUT)
        dummy = self.awg.stop_immediate()

    def run(self):
        self.isRunning = True
        self.sigStatus.emit('Running')
        while self.isRunning:
            if self.current_run > 0 and np.mod(self.current_run, 20) == 0:
                self.acqSaveSum()
            self.sigScanNumber.emit(self.current_run + 1)
            self.acqSingleRun()

        self.sigStatus.emit('Saving')
        self.acqSave()
        self.sigStatus.emit('Done')
        print("Done!")

    def resetCounter(self):
        self.current_run = 0

    def config(self,
               totNumSegments,
               numSegments,
               filename,
               awg,
               scope,
               digi,
               marker_channel,
               marker_point,
               segmented_mode = True
              ):
        self.totNumSegments = totNumSegments
        self.numSegments = numSegments
        self.filename = filename
        self.awg = awg
        if scope is not None:
            self.scope = scope
            self.digi = False
        elif digi is not None:
            self.scope = digi
            self.digi = True
        else:
            print("Warning: Digitizer or Scope not connected")
            return

        self.segmented_mode = segmented_mode
        if marker_channel in [1,2,3,4]:
            self.marker_channel = marker_channel
        else:
            self.marker_channel = None
        self.marker_point = marker_point

    def acqSingleRun(self):

        # acquire data
        print('Acquire Data!')
        if self.digi:
            self.acquireDataDigi()
        else:
            self.acquireData()
        print('Acquisition Done!')

        print(probe.keys())
        for channel in probe.keys():
            fname =  self.filename + '_ch%d_%d' % (channel,
                                                   self.current_run)
            print(str(fname))

            if self.segmented_mode:
                probe[channel] = \
                        self.scope.acquire_segments_multipulse(
                            None, #str(fname),
                            self.totNumSegments,
                            self.numSegments,
                            start = None,
                            size = None,
                            signal_channel = channel,
                            probe_channel = self.marker_channel,
                            probe_point = self.marker_point
                        )
                for j in range(self.numSegments):
                    probe_avg[channel][j].y += probe[channel][j].y
            else:
                # average mode
                probe[channel][0] = \
                        self.scope.acquire_avg(
                            #str(fname), 
                            signal_channel = channel)
                        
                probe_avg[channel][0].y += probe[channel][0].y
        self.current_run += 1
        print('Transfer Data Done!')

        self.sigUpdatePlots.emit()
        print('Plots updated')

    def stop(self):
        self.isRunning = False
        self.sigStatus.emit('Stopping')

    
    def acqSaveSum(self):
 
        # Obtain average from sum and generate Average and Difference
        for channel in probe.keys():
            for j in range(self.numSegments):
                # save averaged spectrum
                fname = '%s_ch%d_probe%d_sum%d' % \
                        (self.filename, 
                         channel, 
                         j,
                         self.current_run)
                print(str(fname))
                probe_avg[channel][j].save(str(fname), ftype = 'npy')
        print("Save done!")

       

    def acqSave(self):

        # Obtain average from sum and generate Average and Difference
        for channel in probe.keys():
            for j in range(self.numSegments):
                probe_avg[channel][j].y = probe_avg[channel][j].y\
                        / self.current_run
                # save averaged spectrum
                fname = '%s_ch%d_probe%d_avg%d' % \
                        (self.filename, 
                         channel, 
                         j,
                         self.current_run)
                print(str(fname))
                probe_avg[channel][j].save(str(fname), ftype = 'npy')
        print("Save done!")


class FFTThread(QtCore.QThread):

    result = QtCore.pyqtSignal(object, object)

    def __init__(self, 
                 parent = None):
		"""
		Initializes the thread.

		"""
#                print("Thread init")
		QtCore.QThread.__init__(self)

    def setControlParameter(self, 
                            tstart = None,
                            tstop = None,
                            unit = 'V',
                            isComplexFFT = True,
                            window_function = 'Boxcar',
                            samplerate = None
                           ):
        self.tstart = tstart
        self.tstop = tstop
        self.unit = unit
        self.isComplexFFT = isComplexFFT,
        self.window_function = window_function
        self.samplerate = samplerate

    def setLoFreq(self, lofreq = 0.0):
        self.lo_freq = lofreq

    def setData(self,
                taxis = None,
                iaxis = None,
                qaxis = None):
        self.taxis = taxis
        self.iaxis = iaxis
        self.qaxis = qaxis
        
        if self.tstart is not None:
            xmin = np.abs(self.taxis - self.tstart).argmin()
            self.taxis = self.taxis[xmin:]
            if len(self.iaxis)>0:
                self.iaxis = self.iaxis[xmin:]
            if len(self.qaxis)>0:
                self.qaxis = self.qaxis[xmin:]


        if self.tstop is not None:
            xmax = np.abs(self.taxis - self.tstop).argmin()
            if len(self.iaxis)>0:
                self.iaxis = self.iaxis[:xmax]
            if len(self.qaxis)>0:
                self.qaxis = self.qaxis[:xmax]

        self.N = len(iaxis)
        self.zero_filling = np.power(2,int(np.ceil(np.log2(self.N))))

    def calcFFT(self):
        try:
            if self.unit == 'Volt':
                if self.isComplexFFT:
                    ty = self.iaxis + 1.0j * self.qaxis
                else:
                    ty = self.iaxis
#
                try:
                    x, y = spec.calc_amplitude_spec_win(ty, 
                                                self.samplerate,
                                                self.window_function,
                                                self.zero_filling
                                               )
                except Exception as e:
                    print("Error calling calc",e)
                try:
                    self.result.emit(x,y)
                except Exception as e:
                    print("Error emit",e)

            else:
                if self.isComplexFFT:
                    ty = self.iaxis + 1.0j * self.qaxis
                else:
                    ty = self.iaxis
                x, y = spec.calc_power_spec_win(ty, 
                                                self.samplerate,
                                                self.window_function,
                                                self.zero_filling
                                               )
                self.result.emit(x,y)

        except Exception as e:
            print("FFT Error", e)

    def run(self):
        if self.N > 0:
            self.calcFFT()


# determine the correct containing the *.ui files
ui_filename = 'ChirpMainWindow.ui'
ui_path = ""
for p in (os.path.dirname(os.path.realpath(__file__)),
		  os.path.dirname(__file__),
		  os.path.dirname(os.path.realpath(sys.argv[0]))):
	if os.path.isfile(os.path.join(p, ui_filename)):
		ui_path = p # should be most reliable, even through symlinks
		break
	elif os.path.isfile(os.path.join(p, "resources", ui_filename)):
		ui_path = os.path.join(p, "resources") # should be most reliable, even through symlinks
		break
if ui_path == "":
	raise IOError("could not identify the *.ui files")

Ui_MainWindow, QMainWindow = loadUiType(os.path.join(ui_path, ui_filename))
class ChirpGUI(QMainWindow, Ui_MainWindow):
	"""
	Defines the main window of the GUI.
	"""
        sigControlParameter = QtCore.pyqtSignal(float, float, str, bool, str,
                                                float)
        sigSendData = QtCore.pyqtSignal(object, object, object)
        sigSetLoFreq = QtCore.pyqtSignal(float)

        sigAcqConfigure = QtCore.pyqtSignal(int, int, str, object, object,
                                            object, int, int, bool)
        sigAcqResetCounter = QtCore.pyqtSignal()

	signalShowMessage = QtCore.pyqtSignal(str, list)
	signalDisableInstrument = QtCore.pyqtSignal(str)
	signalPressuresUpdateGUI = QtCore.pyqtSignal(str, str, str)
	signalMFCUpdateGUI = QtCore.pyqtSignal(list)
	signalScanUpdateIterLCD = QtCore.pyqtSignal(int)
	signalScanStart = QtCore.pyqtSignal(int, str, str)
	signalScanStop = QtCore.pyqtSignal(bool, bool)
	signalScanSave = QtCore.pyqtSignal(str, str)
	signalScanUpdateInfo = QtCore.pyqtSignal()
	signalScanUpdatePlot = QtCore.pyqtSignal()
	signalMonUpdatePlot = QtCore.pyqtSignal()
	def __init__(self):
		"""
		All the functionality of the main buttons should be defined here,
		as well as additional initializations (e.g. the plot window) and
		looping timers.
		"""
		# super is basically mandatory; see
		# http://stackoverflow.com/questions/23981625/why-is-super-used-so-much-in-pyside-pyqt
		# for more info
		super(self.__class__, self).__init__()
		self.setupUi(self)	# was defined in a skeleton automatically
		self.timeGUIStart = datetime.datetime.now()
		self.programName = "Chirp GUI"
		getGitHashCmd = ['git',
			'-C', os.path.dirname(os.path.dirname(os.path.realpath(sys.argv[0]))),
			'rev-parse', '--short', 'HEAD']
#		self.programGitHash = subprocess.check_output(getGitHashCmd).strip()
#		self.setWindowIcon(QtGui.QIcon('casac_icon.png'))
		self.debugging = False

		### misc parameters
#		self.sensorSaveFile = ""
		# to btn_AdvSettingsFilesInstruments
		self.defaultSaveFile = "~/.chirp_gui.conf"
		self.saveperiodSensorData = 30000 # [ms]
		self.updateperiodCheckTab = 500 # [ms]
		self.updateperiodCheckInstruments = 200 # [ms]
		self.updateperiodPressureUpdate = 500 # [ms]
		self.updateperiodTemperatureUpdate = 500 # [ms]
		self.updateperiodMFCUpdate = 500 # [ms]
		# to btn_AdvSettingsMFC
#		self.serialPortMFC = "/dev/ttyS2"
		# to btn_AdvSettingsSynth
		self.reasonableMaxFreqPoints = 100000
		self.reasonableMaxScanTime = 30 # scan time in [min]
		self.synthFreqResolution = 3 # number of decimals points in [Hz]
		self.maxRFInputStd = 10.0 # [dBm]
		self.maxRFInputHi = 0.0 # [dBm]
		self.RFstepPow = 0.01 # dBm
		self.RFstepTime = 0.005 # [s]
		# to btn_AdvSettingsLockin
		self.waitForAutoPhase = 0.05 # [s]
		self.maxLockinFreq = 102000 # [Hz]
		# to btn_AdvSettingsPlot
		self.waitBeforeScanRestart = 0.05 # [s]
		self.updateperiodScanUpdateSlow = 500 # [ms]
		self.updateperiodPlotUpdateFast = 50 # [ms]
		self.updateperiodPlotUpdateSlow = 200 # [ms]
		self.spikeFilterThreshold = 10 # int

		### real-time statuses
                self.hasConnectedScope = False
                self.hasConnectedDigitizer = False
                self.hasConnectedAWG = False
                self.hasConnectedSynth = False

		self.hasConnectedPressure1 = False
		self.hasConnectedPressure2 = False
		self.hasConnectedTemperature = False
		self.hasConnectedMFC = False
		self.hasConnectedSynth = False
		self.hasConnectedLockin = False
		self.isMFCTab = False
		self.isScanTab = False
		self.isMonTab = False
		self.isScanning = False
		self.isPaused = False
		self.isBatchRunning = False
		self.isBatchReadyForNext = False
		self.isScanDirectionUp = True
		self.currentScanIndex = 0
		self.hasPermissionHigherRF = False
		self.scanFinished = False

		### misc items
		self.guiParamsAll = []
		self.guiParamsCollected = []
		self.pressureData = []
		self.temperatureData = []
		self.MFCData = []
		self.sensorSaveFileFH = 0
		self.freqList = []
		self.freqDelay = 0
		self.scanYvalsAvg = []
		self.timeScanStart = 0
		self.timeScanStop = 0
		self.timeScanSave = 0
		self.timeScanPaused = datetime.timedelta(seconds=0)
		self.freqFormattedPrecision = "%f"
		self.scanPlotLabels = []
		self.scanPlotRegion = None
		self.tickFont = QtGui.QFont()
		self.tickFont.setPixelSize(16)

		### define sockets to instruments
		self.socketPressure1 = None
		self.socketPressure2 = None
		self.socketTemperature = None
		self.socketMFC = None
		self.socketSynth = None
		self.socketLockin = None


                ### define settings for the measurement /acquisition
                self.measurement_settings = Settings()
                self.measurement_settings.signal_channels = {}


                self.lr1 = pg.LinearRegionItem([0,0])
                self.lr1.setZValue(-10)
                self.lr2 = pg.LinearRegionItem([0,0])
                self.lr2.setZValue(-10)

                self.lr1.sigRegionChanged.connect(self.lr1Update)
                self.lr2.sigRegionChanged.connect(self.lr2Update)

		### button functionalities
#		# main buttons
#		# file buttons
                self.pB_WorkDirectory.clicked.connect(self.setWorkDirectory)
                self.pB_Filename.clicked.connect(self.setFilename)
#		# instrument buttons
                self.pB_ScopeConnect.clicked.connect(self.connectScope)
                self.pB_ScopeDisconnect.clicked.connect(self.disconnectScope)
                self.pB_AWGConnect.clicked.connect(self.connectAWG)
                self.pB_AWGDisconnect.clicked.connect(self.disconnectAWG)
                self.pB_DigitizerConnect.clicked.connect(self.connectDigitizer)
                self.pB_DigitizerDisconnect.clicked.connect(self.disconnectDigitizer)
#		self.btn_MFCConnect.clicked.connect(self.connectMFC)
		self.pB_SynthConnect.clicked.connect(self.connectSynth)
#		self.btn_LockinConnect.clicked.connect(self.connectLockin)
#		self.btn_MFCDisconnect.clicked.connect(self.disconnectMFC)
		self.pB_SynthDisconnect.clicked.connect(self.disconnectSynth)
#		self.btn_LockinDisconnect.clicked.connect(self.disconnectLockin)
                # awg buttons
                self.pB_AWGRead.clicked.connect(self.readAWGSettings)
                self.pB_AWGSet.clicked.connect(self.setAWGSettings)
                # scope buttons
                self.pB_ScopeRead.clicked.connect(self.readScopeSettings)
                self.pB_ScopeSet.clicked.connect(self.setScopeSettings)
                # digitizer buttons
                self.pB_DigitizerRead.clicked.connect(self.readDigitizerSettings)
                self.pB_DigitizerSet.clicked.connect(self.setDigitizerSettings)
#		# pressure buttons
#		# mfc buttons
		# synth buttons
                self.pB_SynthRead.clicked.connect(self.readSynthSettings)
                self.pB_SynthSet.clicked.connect(self.setSynthSettings)
         	# scan buttons

                # FFT Thread
                self.fftThread = FFTThread()
                self.sigControlParameter.connect(self.fftThread.setControlParameter)
                self.sigSendData.connect(self.fftThread.setData)
                self.sigSetLoFreq.connect(self.fftThread.setLoFreq)
                self.fftThread.result.connect(self.updateFFTWindow)

                self.acqThread = AcqThread()
                self.acqThread.sigStatus.connect(self.acqSetStatus)
                self.acqThread.sigScanNumber.connect(self.acqSetScanNumber)
                self.acqThread.sigUpdatePlots.connect(self.updatePlots)
                self.sigAcqConfigure.connect(self.acqThread.config)
                self.sigAcqResetCounter.connect(self.acqThread.resetCounter)
 #               self.sigAcqStop.connect(self.acqThread.stop)

 #               self.pB_StartAcquisition.clicked.connect(self.startAcquisition)
 #               self.pB_StopAcquisition.clicked.connect(self.stopAcquisition)
                self.pB_StartAcquisition.clicked.connect(self.acqInit)
                self.pB_ContinueAcquisition.clicked.connect(self.acqContinue)
                self.pB_StopAcquisition.clicked.connect(self.acqThread.stop)
#                self.pB_StopAcquisition.clicked.connect(self.acqStop)
                self.pB_SetAcquisition.clicked.connect(self.setAcquisition)
                self.pB_AddToPlot.clicked.connect(self.addToPlot)
                self.pB_RemoveFromPlot.clicked.connect(self.removeFromPlot)
                self.sB_SegmentId.valueChanged.connect(self.updatePlots)

                self.pB_FFT.clicked.connect(self.FFTrun)
#
#		### modify/redefine certain GUI elements
#		self.txt_MFC1FSabs.opts['formatString'] = "%.2f"
#		self.txt_MFC2FSabs.opts['formatString'] = "%.2f"
#		self.txt_MFC3FSabs.opts['formatString'] = "%.2f"
#		self.txt_MFC4FSabs.opts['formatString'] = "%.2f"
#		self.txt_MFC1FSabs.opts['constStep'] = 1
#		self.txt_MFC2FSabs.opts['constStep'] = 1
#		self.txt_MFC3FSabs.opts['constStep'] = 1
#		self.txt_MFC4FSabs.opts['constStep'] = 1
#		self.txt_SynthFreqStart.opts['formatString'] = "%.3f"
#		self.txt_SynthFreqStart.opts['constStep'] = 1
#		self.txt_SynthFreqEnd.opts['formatString'] = "%.3f"
#		self.txt_SynthFreqEnd.opts['constStep'] = 1
#		self.txt_SynthFreqStep.opts['formatString'] = "%.4f"
#		self.txt_SynthFreqStep.opts['relStep'] = 50
#		self.txt_SynthDelay.opts['formatString'] = "%d"
#		self.txt_SynthDelay.opts['constStep'] = 1
#		self.txt_SynthEZFreqCenter.opts['formatString'] = "%.3f"
#		self.txt_SynthEZFreqCenter.opts['constStep'] = 1
#		self.txt_SynthEZFreqSpan.opts['formatString'] = "%.1f"
#		self.txt_SynthEZFreqSpan.opts['constStep'] = 1
#		self.txt_SynthAMFreq.opts['formatString'] = "%.1f"
#		self.txt_SynthAMFreq.opts['constStep'] = 5
#		self.txt_SynthFMFreq.opts['formatString'] = "%.1f"
#		self.txt_SynthFMFreq.opts['constStep'] = 0.5
#		self.txt_SynthFMWidth.opts['formatString'] = "%d"
#		self.txt_SynthFMWidth.opts['constStep'] = 50
#		self.txt_LockinPhase.opts['formatString'] = "%.1f"
#		self.txt_LockinPhase.opts['constStep'] = 1
#		self.txt_MonFreq.opts['formatString'] = "%.3f"
#		self.txt_MonFreq.opts['constStep'] = 1
#
#		### define the contents of various comboboxes

                self.awgSamplerates = [
                    ('12 GSa/s', 12.0e9),
                ]
                for sr in self.awgSamplerates:
                    self.coB_AWGSamplerate.addItem(sr[0],sr[1])

                self.awgCurrentModes = [
                    ('Arbitrary', 'ARB'),
                    ('STSequence', 'STS'),
                    ('STScenario', 'STSC'),
                ]
                for mode in self.awgCurrentModes:
                    self.coB_AWGCurrentMode.addItem(mode[0], mode[1])

                # coB_AWGSelectedOutCH1 ('AC', 'DC') 
                # coB_AWGSelectedOutCH2 ('AC', 'DC')

                self.awgArmingModes = [
                    ('self', 'SELF'),
                    ('Armed', 'ARM'),
                ]
                for mode in self.awgArmingModes:
                    self.coB_AWGArmingModeCH1.addItem(mode[0], mode[1])
                    self.coB_AWGArmingModeCH2.addItem(mode[0], mode[1])

                self.awgTriggerModes = [
                    ('Continuous', 'Continuous'),
                    ('Triggered', 'Triggered'),
                    ('Gated', 'Gated'),
                ]

                for mode in self.awgTriggerModes:
                    self.coB_AWGTriggerModeCH1.addItem(mode[0], mode[1])
                    self.coB_AWGTriggerModeCH2.addItem(mode[0], mode[1])

                self.awgTriggerSources = [
                    ('Internal', 'INT'),
                    ('External', 'EXT'),
                ]
                for source in self.awgTriggerSources:
                    self.coB_AWGTriggerSource.addItem(source[0], source[1])

                self.awgOscillators = [
                    ('Internal', 'INT'),
                    ('External', 'EXT'),
                    ('AXI Backplane', 'AXI'),
                ]
                for osc in self.awgOscillators:
                    self.coB_AWGReferenceClock.addItem(osc[0], osc[1])


                self.scopeSamplerates = [
                    ('80 GSa/s', 80.0e9),
                    ('40 GSa/s', 40.0e9),
                    ('20 GSa/s', 20.0e9),
                    ('10 GSa/s', 10.0e9),
                    ('5 GSa/s', 5.0e9),
                    ('4 GSa/s', 4.0e9),
                    ('2.5 GSa/s', 2.5e9),
                    ('2.0 GSa/s', 2.0e9),
                    ('1.25 GSa/s', 1.25e9),
                    ('1.0 GSa/s', 1.0e9),
                    ('500 MSa/s', 500.0e6),
                    ('400 MSa/s', 400.0e6),
                    ('250 MSa/s', 250.0e6),
                    ('200 MSa/s', 200.0e6),
                    ('125 MSa/s', 125.0e6),
                    ('100 MSa/s', 100.0e6),
                    ('50 MSa/s', 50.0e6),
                    ('25 MSa/s', 25.0e6),
                    ('20 MSa/s', 20.0e6),
                    ('10 MSa/s', 10.0e6),
                ]
                for sr in self.scopeSamplerates:
                    self.coB_ScopeSamplerate.addItem(sr[0],sr[1])

                self.acquisition_modes = [
                    ('Real Time', 'RTIM'),
                    ('High Resolution', 'HRES'),
                    ('Equivalent Time', 'ETIM'),
                    ('Peak Detect', 'PDET'),
                    ('Segmented - Normal', 'SEGM'),
                    ('Segmented - High Resolution', 'SEGH'),
                    ('Segmented - Peak Detect'),
                ]
                for am in self.acquisition_modes:
                    self.coB_ScopeAcquisitionMode.addItem(am[0],am[1])

                self.scope_reference_clocks = [
                    ('Internal', 'INT'),
                    ('External', 'EXT'),
                ]
                for rc in self.scope_reference_clocks:
                    self.coB_ScopeReferenceClock.addItem(rc[0],rc[1])

                self.vertical_units = [
                    ('Volt', 'VOLT'),
                    ('Watt', 'WATT'),
                    ('Ampere', 'AMP'),
                    ('Unknown', 'UNKN'),
                ]
                for unit in self.vertical_units:
                    self.coB_ScopeVerticalUnitCH1.addItem(unit[0],unit[1])
                    self.coB_ScopeVerticalUnitCH2.addItem(unit[0],unit[1])
                    self.coB_ScopeVerticalUnitCH3.addItem(unit[0],unit[1])
                    self.coB_ScopeVerticalUnitCH4.addItem(unit[0],unit[1])

                self.coB_PlotId.addItem('Plot 1', 0)
                self.coB_PlotId.addItem('Plot 2', 1)

                self.fft_sources = [
                    ('Average', 'AVG'),
                    ('Last Scan', 'LS'),
                    ('Segment Average', 'SAVG'),
                ]
                for s in self.fft_sources:
                    self.coB_FFTSource.addItem(s[0], s[1])

                self.window_functions = [
                    'Boxcar',
                    'Hamming',
                    'Hann',
                    'Blackman',
                    'Flattop',
                    'Blackmanharris',
                    'Kaiser',
                    'Triangle',
                    'Tukey',
                    'Barthann',
                    'Barlett'
                ]
                for s in self.window_functions:
                    self.coB_WindowFunction.addItem(s,s)

                self.digitizerSamplerates = [
                    ('5 GSa/s', 5.0e9)
                ]
                for sr in self.digitizerSamplerates:
                    self.coB_DigitizerSamplerate.addItem(sr[0],sr[1])

                self.digitizer_acquisition_modes = [
                    ('Single', 'single'),
                    ('Segmented', 'segmented'),
                    ('Average', 'avg')
                ]
                for am in self.digitizer_acquisition_modes:
                    self.coB_DigitizerAcquisitionMode.addItem(am[0],am[1])

                self.digitizer_reference_oscillator = [
                    ('External', 'EXT'),
                    ('Internal', 'INT')
                ]
                for ra in self.digitizer_reference_oscillator:
                    self.coB_DigitizerReferenceOscillator.addItem(ra[0],ra[1])

                self.digitizer_vertical_range = [
                    ('1.0 V', 1.0),
                    ('250 mV', 0.25)
                ]
                for vr in self.digitizer_vertical_range:
                    self.coB_DigitizerVerticalRange.addItem(vr[0],vr[1])
                self.digitizer_trigger_source = [
                    ('External', 'ext'),
                    ('Internal', 'int')
                ]
                for ts in self.digitizer_trigger_source:
                    self.coB_DigitizerTriggerSource.addItem(ts[0],ts[1])
                self.digitizer_trigger_slope = [
                    ('Positive', 'positive'),
                    ('Negative', 'negative')
                ]
                for ts in self.digitizer_trigger_slope:
                    self.coB_DigitizerTriggerSlope.addItem(ts[0],ts[1])






        def setWorkDirectory(self):
            self.lE_WorkDirectory.setText(QtGui.QFileDialog.getExistingDirectory())

        def setFilename(self):
            self.working_dir = self.lE_WorkDirectory.text()

            if len(self.working_dir) > 0:
                filename, _filter = QtGui.QFileDialog.getSaveFileName(
                    caption = 'Open file', 
                    directory = self.working_dir)
            else:
                filename, _filter = QtGui.QFileDialog.getSaveFileName()
            print(filename)
            print(_filter)
            self.lE_Filename.setText(filename)
#            self.lE_Filename.setText(QtGui.QFileDialog.getSaveFileName())

        def connectSynth(self):
            self.synth_settings = Settings()

            self.synth_settings.frequency = 22000.0 # MHz
            self.synth_settings.power = 0.0 # dBm
            self.synth_settings.host = SYNTHESIZER_HOST
            self.synth_settings.port = SYNTHESIZER_PORT
            
            try:
                self.synth = synthesizer.PSG_signal_generator(self.synth_settings.host,
                                                              self.synth_settings.port,
                                                              com = 'TCPIP')
                self.hasConnectedSynth = True
                self.cB_UseSynthesizer.setChecked(True)
            except:
                self.hasConnectedSynth = False
                self.cB_UseSynthesizer.setChecked(False)
            self.readSynthSettings()

        def disconnectSynth(self):
            self.synth.close()
            self.cB_UseSynthesizer.setChecked(False)

        def readSynthSettings(self):

            self.synth_settings.frequency = self.synth.get_frequency() * 1.0e-6
            self.synth_settings.rf_power_state = self.synth.get_rf_power_state()
            self.dSB_SynthFrequency.setValue(self.synth_settings.frequency)
            self.synth_settings.power_level = self.synth.get_power_level()
            self.dSB_SynthPowerLevel.setValue(self.synth_settings.power_level)
            if self.synth.get_rf_power_state() == 1:
                self.cB_RF_PowerState.setChecked(True)
            else:
                self.cB_RF_PowerState.setChecked(False)

        def setSynthSettings(self):

            self.synth.set_frequency(self.dSB_SynthFrequency.value())
            self.synth.set_power_level(self.dSB_SynthPowerLevel.value())
            if self.cB_RF_PowerState.isChecked():
                self.synth.switch_rf_power('ON')
            else:
                self.synth.switch_rf_power('OFF')

            self.readSynthSettings()

        def connectAWG(self):
            """
            Connect the AWG
            """
            self.awg_settings = Settings()
            self.awg_settings.direct_mode = 'SPEED'

            apply_settings = True

            try:
                self.awg = awglib.AWG(host = AWG_HOST, port = AWG_PORT)
                if apply_settings:
                    self.awg.apply_settings(settings = self.awg_settings)
                    #scope.read_settings()
                self.hasConnectedAWG = True
                self.cB_UseAWG.setChecked(True)
            except Exception as e:
                print(e)
                self.hasConnectedAWG = False
                self.cB_UseAWG.setChecked(False)

            self.readAWGSettings()


        def disconnectAWG(self):
            """
            Disconnect the awg
            """
            self.awg.close()
            self.cB_UseAWG.setChecked(False)


        def readAWGSettings(self):
            """
            Read awg setting from the instrument and pass it to the GUI.
            """
            self.awg.read_settings()
            self.awg.print_settings()
            s = self.awg.settings

            # set GUI elements accordingly
            srate = self.awg.get_samplerate()
            idx = self.coB_AWGSamplerate.findData(srate)
            if idx != -1:
                self.coB_AWGSamplerate.setCurrentIndex(idx)
            idx = self.coB_AWGCurrentMode.findData(s.get('current_mode'))
            if idx != -1:
                self.coB_AWGCurrentMode.setCurrentIndex(idx)
            idx = self.coB_AWGSelectedOutCH1.findText(s.get('selected_out_ch1'))
            if idx != -1:
                self.coB_AWGSelectedOutCH1.setCurrentIndex(idx)
            idx = self.coB_AWGSelectedOutCH2.findText(s.get('selected_out_ch2'))
            if idx != -1:
                self.coB_AWGSelectedOutCH2.setCurrentIndex(idx)

            idx = self.coB_AWGArmingModeCH1.findData(s.get('arming_mode_ch1'))
            if idx != -1:
                self.coB_AWGArmingModeCH1.setCurrentIndex(idx)
            idx = self.coB_AWGArmingModeCH2.findData(s.get('arming_mode_ch2'))
            if idx != -1:
                self.coB_AWGArmingModeCH2.setCurrentIndex(idx)

            
            if int(s.get('amp_output_ch1')) == 1:
                self.cB_AWGOutput1On.setChecked(True)
            else:
                self.cB_AWGOutput1On.setChecked(False)

            if int(s.get('amp_output_ch2')) == 1:
                self.cB_AWGOutput2On.setChecked(True)
            else:
                self.cB_AWGOutput2On.setChecked(False)

            idx = self.coB_AWGTriggerModeCH1.findData(s.get('trigger_mode_ch1'))
            if idx != -1:
                self.coB_AWGTriggerModeCH1.setCurrentIndex(idx)
            idx = self.coB_AWGTriggerModeCH2.findData(s.get('trigger_mode_ch2'))
            if idx != -1:
                self.coB_AWGTriggerModeCH2.setCurrentIndex(idx)

            idx = self.coB_AWGTriggerSource.findData(s.get('trigger_source'))
            if idx != -1:
                self.coB_AWGTriggerSource.setCurrentIndex(idx)
            print("Reference Oscillator:")
            print(s.get('reference_oscillator'))
            idx = self.coB_AWGReferenceClock.findData(s.get('reference_oscillator'))
            if idx != -1:
                self.coB_AWGReferenceClock.setCurrentIndex(idx)


            self.dSB_AWGOutputVoltageCH1.setValue(s.get('output_voltage_ch1'))
            self.dSB_AWGOutputVoltageCH2.setValue(s.get('output_voltage_ch2'))
            
            self.dSB_AWGTriggerFrequency.setValue(s.get('trigger_frequency'))
            self.dSB_AWGOscillatorFrequency.setValue(s.get('oscillator_frequency')
                                                     / 1.0e6 )
            self.dSB_AWGSync1Amplitude.setValue(s.get('sync1_marker_amplitude'))
            self.dSB_AWGSync2Amplitude.setValue(s.get('sync2_marker_amplitude'))
            self.dSB_AWGSample1Amplitude.setValue(s.get('samp1_marker_amplitude'))
            self.dSB_AWGSample2Amplitude.setValue(s.get('samp2_marker_amplitude'))
            self.dSB_AWGSync1Offset.setValue(s.get('sync1_marker_offset'))
            self.dSB_AWGSync2Offset.setValue(s.get('sync2_marker_offset'))
            self.dSB_AWGSample1Offset.setValue(s.get('samp1_marker_offset'))
            self.dSB_AWGSample2Offset.setValue(s.get('samp2_marker_offset'))


        def setAWGSettings(self):
            """
            Set values from GUI
            """
            self.awg.read_settings()
            s = self.awg.settings
            s.set('samplerate', 
                  self.coB_AWGSamplerate.currentData())

            s.set('current_mode',
                  self.coB_AWGCurrentMode.currentData())
            s.set('selected_out_ch1',
                  self.coB_AWGSelectedOutCH1.currentText())
            s.set('selected_out_ch2',
                  self.coB_AWGSelectedOutCH2.currentText())
            s.set('arming_mode_ch1',
                  self.coB_AWGArmingModeCH1.currentData())
            s.set('arming_mode_ch2',
                  self.coB_AWGArmingModeCH2.currentData())

            if self.cB_AWGOutput1On.isChecked():
                s.set('amp_output_ch1',1)
            else:
                s.set('amp_output_ch1',0)
            if self.cB_AWGOutput2On.isChecked():
                s.set('amp_output_ch2',1)
            else:
                s.set('amp_output_ch2',0)

            s.set('trigger_mode_ch1', 
                  self.coB_AWGTriggerModeCH1.currentData())
            s.set('trigger_mode_ch2',
                  self.coB_AWGTriggerModeCH2.currentData())
            s.set('trigger_source',
                  self.coB_AWGTriggerSource.currentData())
            s.set('reference_oscillator',
                  self.coB_AWGReferenceClock.currentData())


            s.set('output_voltage_ch1',
                  self.dSB_AWGOutputVoltageCH1.value())
            s.set('output_voltage_ch2',
                  self.dSB_AWGOutputVoltageCH2.value())
            s.set('trigger_frequency',
                  self.dSB_AWGTriggerFrequency.value())
            s.set('oscillator_frequency',
                  self.dSB_AWGOscillatorFrequency.value() * 1.0e6)
            s.set('sync1_marker_amplitude',
                  self.dSB_AWGSync1Amplitude.value())
            s.set('sync2_marker_amplitude',
                  self.dSB_AWGSync2Amplitude.value())
            s.set('samp1_marker_amplitude',
                  self.dSB_AWGSample1Amplitude.value())
            s.set('samp2_marker_amplitude',
                  self.dSB_AWGSample2Amplitude.value())
            s.set('sync1_marker_offset',
                  self.dSB_AWGSync1Offset.value())
            s.set('sync2_marker_offset',
                  self.dSB_AWGSync2Offset.value())
            s.set('samp1_marker_offset',
                  self.dSB_AWGSample1Offset.value())
            s.set('samp2_marker_offset',
                  self.dSB_AWGSample2Offset.value())

            self.awg.apply_settings(settings = s)
            self.awg.read_settings()




        def connectScope(self):
            """
            Connect the scope
            """
            self.scope_settings = Settings()
            self.scope_settings.byte_order = 'LSBF'
            self.scope_settings.encoding = 'WORD'

            apply_settings = True

            try:
                self.scope = scopelib.initScope()
                if apply_settings:
                    self.scope.apply_settings(settings = self.scope_settings)
                    #scope.read_settings()
                self.hasConnectedScope = True
                self.cB_UseScope.setChecked(True)
            except:
                self.hasConnectedScope = False
                self.cB_UseScope.setChecked(False)

            self.readScopeSettings()


        def disconnectScope(self):
            """
            Disconnect the scope
            """
            self.scope.close()
            self.cB_UseScope.setChecked(False)

        def readScopeSettings(self):
            """
            Read scope setting from the instrument and pass it to the GUI.
            """
            self.scope.read_settings()
            self.scope.print_settings()
            s = self.scope.settings

            # set GUI elements accordingly
            srate = self.scope.get_samplerate()
            idx = self.coB_ScopeSamplerate.findData(srate)
            if idx != -1:
                self.coB_ScopeSamplerate.setCurrentIndex(idx)
            timebase_range = s.get('horizontal_range') * 1.0e6 
            timebase_position = s.get('horizontal_position') * 1.0e6
            self.dSB_ScopeHorizontalRange.setValue(timebase_range)
            self.dSB_ScopeHorizontalPosition.setValue(timebase_position)
            
            idx = self.coB_ScopeAcquisitionMode.findData(s.get('acquisition_mode'))
            self.coB_ScopeAcquisitionMode.setCurrentIndex(idx)

            self.cB_ScopeAverageMode.setChecked(s.get('average_mode'))
            self.sB_ScopeNumAverages.setValue(s.get('num_averages'))
            idx = self.coB_ScopeReferenceClock.findData(s.get('reference_clock'))
            self.coB_ScopeReferenceClock.setCurrentIndex(idx)
            self.dSB_ScopeVerticalRangeCH1.setValue(s.get('vertical_range_ch1'))
            self.dSB_ScopeVerticalRangeCH2.setValue(s.get('vertical_range_ch2'))
            self.dSB_ScopeVerticalRangeCH3.setValue(s.get('vertical_range_ch3'))
            self.dSB_ScopeVerticalRangeCH4.setValue(s.get('vertical_range_ch4'))
#vertical_scale_ch1: 0.001
#vertical_scale_ch2: 0.2
#vertical_scale_ch3: 0.002
#vertical_scale_ch4: 0.2
            idx = self.coB_ScopeVerticalUnitCH1.findData(s.get('vertical_unit_ch1'))
            self.coB_ScopeVerticalUnitCH1.setCurrentIndex(idx)
            idx = self.coB_ScopeVerticalUnitCH2.findData(s.get('vertical_unit_ch2'))
            self.coB_ScopeVerticalUnitCH2.setCurrentIndex(idx)
            idx = self.coB_ScopeVerticalUnitCH3.findData(s.get('vertical_unit_ch3'))
            self.coB_ScopeVerticalUnitCH3.setCurrentIndex(idx)
            idx = self.coB_ScopeVerticalUnitCH4.findData(s.get('vertical_unit_ch4'))
            self.coB_ScopeVerticalUnitCH4.setCurrentIndex(idx)
            self.cB_Channel1.setChecked(s.get('display_chan1'))
            self.cB_Channel2.setChecked(s.get('display_chan2'))
            self.cB_Channel3.setChecked(s.get('display_chan3'))
            self.cB_Channel4.setChecked(s.get('display_chan4'))

        def setScopeSettings(self):
            """
            Set values from GUI
            """
            self.scope.read_settings()
            s = self.scope.settings
            s.set('Samplerate', 
                  self.coB_ScopeSamplerate.currentData())
            s.set('acquisition_mode', 
                  self.coB_ScopeAcquisitionMode.currentData())


            s.set('horizontal_range', 
                  self.dSB_ScopeHorizontalRange.value()/1.0e6 )
            s.set('horizontal_position', 
                  self.dSB_ScopeHorizontalPosition.value()/1.0e6 )
            s.set('average_mode', self.cB_ScopeAverageMode.isChecked())
            s.set('num_averages', 
                  self.sB_ScopeNumAverages.value())
            s.set('reference_clock',
                  self.coB_ScopeReferenceClock.currentData())
            s.set('vertical_range_ch1',
                  self.dSB_ScopeVerticalRangeCH1.value())
            s.set('vertical_range_ch2',
                  self.dSB_ScopeVerticalRangeCH2.value())
            s.set('vertical_range_ch3',
                  self.dSB_ScopeVerticalRangeCH3.value())
            s.set('vertical_range_ch4',
                  self.dSB_ScopeVerticalRangeCH4.value())

            s.set('vertical_unit_ch1',
                  self.coB_ScopeVerticalUnitCH1.currentData())
            s.set('vertical_unit_ch2',
                  self.coB_ScopeVerticalUnitCH2.currentData())
            s.set('vertical_unit_ch3',
                  self.coB_ScopeVerticalUnitCH3.currentData())
            s.set('vertical_unit_ch4',
                  self.coB_ScopeVerticalUnitCH4.currentData())


            s.set('display_chan1',
                  self.cB_Channel1.isChecked())
            s.set('display_chan2',
                  self.cB_Channel2.isChecked())
            s.set('display_chan3',
                  self.cB_Channel3.isChecked())
            s.set('display_chan4',
                  self.cB_Channel4.isChecked())

            self.scope.apply_settings(settings = s)
            self.scope.read_settings()



        def connectDigitizer(self):
            """
            Connect the scope
            """
            #self.digi_settings = Settings()

            #apply_settings = True
            num_averages = 1
            try:
                self.digi = digitizer.Digitizer()
                self.digi.set_trigger_source('ext')
                self.digi.set_vertical_range(1.0)
                self.digi.set_acquisition_mode(mode='avg')
                self.digi.set_num_acquisitions(num_acquisitions=num_averages)
                self.digi.configure_acquisition()
                self.digi.calibrate()

                self.hasConnectedDigitizer = True
                self.cB_UseDigitizer.setChecked(True)
            except Exception as e:
                print("Error: %s" % e)
                self.hasConnectedDigitizer = False
                self.cB_UseDigitizer.setChecked(False)

            self.readDigitizerSettings()


        def disconnectDigitizer(self):
            """
            Disconnect the scope
            """
            self.digi.close()
            self.cB_UseDigitizer.setChecked(False)

        def readDigitizerSettings(self):
            """
            Read scope setting from the instrument and pass it to the GUI.
            """
            #self.digi.read_settings()
            #srate = self.digi.get_samplerate()
            refsource = self.digi.get_reference_source()
            errOnOverflow = self.digi.get_error_on_overflow()
            self.digi.print_settings()

            #self.digi.print_settings()
            s = self.digi.settings

            # set GUI elements accordingly
            idx = self.coB_DigitizerSamplerate.findData(
                s.get('samplerate')
            )
            if idx != -1:
                self.coB_DigitizerSamplerate.setCurrentIndex(idx)
            #timebase_range = s.get('horizontal_range') * 1.0e6 
            #timebase_position = s.get('horizontal_position') * 1.0e6
            #self.dSB_ScopeHorizontalRange.setValue(timebase_range)
            #self.dSB_ScopeHorizontalPosition.setValue(timebase_position)
            
            idx = self.coB_DigitizerAcquisitionMode.findData(s.get('acquisition_mode'))
            self.coB_DigitizerAcquisitionMode.setCurrentIndex(idx)

            #self.cB_ScopeAverageMode.setChecked(s.get('average_mode'))
            self.sB_DigitizerNumAcquisitions.setValue(s.get('num_acquisitions'))
            idx = self.coB_DigitizerReferenceOscillator.findData(s.get('reference_source'))
            self.coB_DigitizerReferenceOscillator.setCurrentIndex(idx)
            idx = self.coB_DigitizerVerticalRange.findData(s.get('vertical_range'))
            self.coB_DigitizerVerticalRange.setCurrentIndex(idx)
            self.dSB_DigitizerRecordLength.setValue(s.get('record_length_time')
                                                    * 1.0e6)
            print(s.get('record_length_time'))
            print(s.get('record_size'))
            self.dSB_DigitizerTriggerLevel.setValue(
                s.get('trigger_level'))
            idx = self.coB_DigitizerTriggerSource.findData(
                s.get('trigger_source'))
            self.coB_DigitizerTriggerSource.setCurrentIndex(idx)
            idx = self.coB_DigitizerTriggerSlope.findData(
                s.get('trigger_slope'))
            self.coB_DigitizerTriggerSlope.setCurrentIndex(idx)
            self.cB_DigitizerErrorOnOverflow.setChecked(errOnOverflow)


        def setDigitizerSettings(self):
            """
            Set values from GUI
            """
            #self.digi.read_settings()
            s = self.digi.settings
            s.set('Samplerate', 
                  self.coB_DigitizerSamplerate.currentData())
            s.set('acquisition_mode', 
                  self.coB_DigitizerAcquisitionMode.currentData())


#            s.set('horizontal_range', 
#                  self.dSB_ScopeHorizontalRange.value()/1.0e6 )
#            s.set('horizontal_position', 
#                  self.dSB_ScopeHorizontalPosition.value()/1.0e6 )
#            s.set('average_mode', self.cB_ScopeAverageMode.isChecked())
            s.set('num_acquisitions', 
                  self.sB_DigitizerNumAcquisitions.value())
            s.set('reference_source',
                  self.coB_DigitizerReferenceOscillator.currentData())
            s.set('vertical_range',
                  self.coB_DigitizerVerticalRange.currentData())
            s.set('record_length_time',
                  self.dSB_DigitizerRecordLength.value() * 1.0e-6)

            s.set('error_on_overflow',
                  self.cB_DigitizerErrorOnOverflow.isChecked())

            s.set('trigger_source',
                  self.coB_DigitizerTriggerSource.currentData())
            s.set('trigger_slope',
                  self.coB_DigitizerTriggerSlope.currentData())
            s.set('trigger_level',
                  self.dSB_DigitizerTriggerLevel.value())

            self.digi.apply_settings(settings = s)
#            self.scope.read_settings()


# -----------------------------------------
# Measurement
# -----------------------------------------

        def setAcquisition(self):
            """
            Set parameters for acquisition
            """
            # Determine mode (avg or seg) and instrument
            if self.hasConnectedDigitizer:
                acqmode = self.digi.settings.get('acquisition_mode')
                if acqmode == 'segmented':
                    self.measurement_settings.segmented_mode = True
                else:
                    self.measurement_settings.segmented_mode = False
                self.measurement_settings.detection = 'digitizer'

            elif self.hasConnectedScope:
                acqmode = self.scope.settings.get('acquisition_mode')
                if acqmode in ('SEGH', 'SEGM'):
                    self.measurement_settings.segmented_mode = True
                else:
                    self.measurement_settings.segmented_mode = False
                self.measurement_settings.detection = 'scope'
            else:
                print("Connect Digitizer or Scope!")
                return

            # Configure Channels to acquire
            self.measurement_settings.signal_channels = []
            self.measurement_settings.display_plot = {}
            if self.cB_AcquireChannel1.isChecked():
                self.measurement_settings.signal_channels.append(1)
                self.measurement_settings.display_plot['CH%d' % 1] = {'plot':0,
                                                                      'showLastScan':True}
            if self.cB_AcquireChannel2.isChecked():
                self.measurement_settings.signal_channels.append(2)
                self.measurement_settings.display_plot['CH%d' % 2] = {'plot':1,
                                                                      'showLastScan':True}
            if self.cB_AcquireChannel3.isChecked():
                self.measurement_settings.signal_channels.append(3)
                self.measurement_settings.display_plot['CH%d' % 3] = {'plot':1,
                                                                      'showLastScan':True}
            if self.cB_AcquireChannel4.isChecked():
                self.measurement_settings.signal_channels.append(4)
                self.measurement_settings.display_plot['CH%d' % 4] = {'plot':1,
                                                                      'showLastScan':True}
            self.measurement_settings.marker_channel = 0
            if self.cB_MarkerChannel1.isChecked():
                self.measurement_settings.marker_channel = 1
            if self.cB_MarkerChannel2.isChecked():
                self.measurement_settings.marker_channel = 2
            if self.cB_MarkerChannel3.isChecked():
                self.measurement_settings.marker_channel = 3
            if self.cB_MarkerChannel4.isChecked():
                self.measurement_settings.marker_channel = 4
            self.measurement_settings.marker_point = self.sB_MarkerPoint.value()


            # Configure segments
            self.measurement_settings.num_loops =\
                    self.sB_NumLoops.value()
            self.measurement_settings.num_segments = \
                    self.sB_NumSegments.value()
            self.measurement_settings.totNumSegments = \
                    self.measurement_settings.num_loops \
                    * self.measurement_settings.num_segments
            if self.measurement_settings.detection == 'scope':
                self.scope.set_num_segments(self.measurement_settings.totNumSegments)
                # test if it works
                totNumSegments = self.scope.get_num_segments_requested()
                # reduce number of loops if number of segments is too large
                if totNumSegments < \
                   self.measurement_settings.totNumSegments:
                    self.measurement_settings.num_loops = \
                            int(totNumSegments /
                                self.measurement_settings.num_segments)
                    self.sB_NumLoops.setValue(self.measurement_settings.num_loops)
                    self.setAcquisition()
            else:
                if acqmode == 'segmented':
                    print("Tot Num Segments: %d" % self.measurement_settings.totNumSegments)
                    self.digi.set_num_acquisitions(self.measurement_settings.totNumSegments)
                    for ch in self.measurement_settings.signal_channels:
                        self.digi.configure_acquisition(channel = 'Channel%d' % ch)
                    self.digi.calibrate()
                    # test if it works
                    totNumSegments = self.digi.get_num_acquisitions()
                    print("Got Tot Num Segments: %d" % totNumSegments)
                    # reduce number of loops if number of segments is too large
                    if totNumSegments < \
                       self.measurement_settings.totNumSegments:
                        self.measurement_settings.num_loops = \
                                int(totNumSegments /
                                    self.measurement_settings.num_segments)
                        self.sB_NumLoops.setValue(self.measurement_settings.num_loops)
                        self.setAcquisition()
                else:
                    print("Tot Num Segments: %d" % self.measurement_settings.totNumSegments)
                    self.digi.set_num_acquisitions(self.measurement_settings.num_loops)
                    for ch in self.measurement_settings.signal_channels:
                        self.digi.configure_acquisition(channel = 'Channel%d' % ch)
                    self.digi.calibrate()
                    # test if it works
                    numSegments = self.digi.get_num_acquisitions()
                    numAvg = self.digi.get_num_averages()
                    print("Got Num Segments: %d" % numSegments)
                    print("Got Num Averages: %d" % numAvg)
                    # reduce number of loops if number of segments is too large
                    if numSegments * numAvg < \
                       self.measurement_settings.totNumSegments:
                        self.measurement_settings.num_loops = \
                                int(np.round( 0.5 * self.measurement_settings.num_loops
                                            ))
                        self.sB_NumLoops.setValue(self.measurement_settings.num_loops)
                        self.setAcquisition()

            self.lE_TotNumSegments.setText('%d' % \
                                           self.measurement_settings.totNumSegments)

            # Configure Channel combo - box
            self.coB_AddChannel.clear()
            for ch in self.measurement_settings.signal_channels:
                self.coB_AddChannel.addItem('Channel%d' % ch, 'CH%d' % ch)

            # initialize signals
            if self.measurement_settings.detection == 'scope':
                self.acqRecordlength = self.scope.get_recordlength()
                self.timebase_range = self.scope.get_timebase_range()
                self.refPoint = self.scope.get_timebase_reference_percent()
                print("Reference Position: %lf" % self.refPoint)
                self.acqXorigin = self.scope.get_timebase_position() -  0.5 * self.timebase_range 
                self.acqXfac = self.scope.get_timebase_range() \
                        / self.acqRecordlength
            else:
                self.acqRecordlength = self.digi.settings.get('record_size')
#                self.timebase_range = self.scope.get_timebase_range()
#                self.refPoint = self.scope.get_timebase_reference_percent()
#                print("Reference Position: %lf" % self.refPoint)
                self.acqXorigin = 0 
                self.acqXfac = 1.0 / self.digi.settings.get('samplerate')


        def addToPlot(self):
            """
            Displays the selected channel in the selected plot-window.
            """
            ch = self.coB_AddChannel.currentData()
            plotid = self.coB_PlotId.currentData()
            showLastScan = self.cB_ShowLastScan.isChecked()
            self.measurement_settings.display_plot[ch]['plot'] = plotid 
            self.measurement_settings.display_plot[ch]['showLastScan'] = \
                    showLastScan
            self.updatePlots()

        def removeFromPlot(self):
            """
            Hides the selected channel.
            """
            ch = self.coB_AddChannel.currentData()
            self.measurement_settings.display_plot[ch]['plot'] = None
            self.updatePlots()

        def updatePlots(self):
            """
            Update the plot widgets and plot data.
            """
            self.FIDWidget.clear()
            self.FID2Widget.clear()
            seg = self.sB_SegmentId.value()
            try:
                num_segs = len(probe_avg[ch].keys())
            except:
                num_segs = 1
            showAVG = self.cB_DisplayAverage.isChecked()
            for sig in self.measurement_settings.display_plot.keys():
                ch = int(sig.replace('CH',''))
                try:
                    plconf = self.measurement_settings.display_plot[sig]
                    if plconf['plot'] == 0:
                        if plconf['showLastScan']:
                            x = probe[ch][seg].x
                            y = probe[ch][seg].y
                            self.FIDWidget.plot(x,y, pen = (ch + 4,12))
                        x = probe_avg[ch][seg].x
                        y = probe_avg[ch][seg].y / self.current_run
                        pl = self.FIDWidget.plot(x,y, pen = (ch, 4))
                        print(pl)
                        if showAVG:
                            y = np.array([probe_avg[ch][s].y 
                                          for s in
                                          probe_avg[ch].keys()]).sum(axis=0)
                            self.FIDWidget.plot(x,y / float(num_segs), pen = (ch + 8,12))
                    elif plconf['plot'] == 1:
                        if plconf['showLastScan']:
                            x = probe[ch][seg].x
                            y = probe[ch][seg].y
                            self.FID2Widget.plot(x,y, pen=(ch + 4,12))
                        x = probe_avg[ch][seg].x
                        y = probe_avg[ch][seg].y / self.current_run
                        pl = self.FID2Widget.plot(x,y, pen = (ch, 4))
                        if showAVG:
                            y = np.array([probe_avg[ch][s].y 
                                          for s in
                                          probe_avg[ch].keys()]).sum(axis=0)
                            self.FID2Widget.plot(x,y / float(num_segs), pen =(ch + 8, 12))
                        print(pl)
                except Exception as e:
                    print(e)

#            lr1 = pg.LinearRegionItem([self.acqXorigin, self.timebase_range])
            self.lr1.setZValue(-10)
#            lr2 = pg.LinearRegionItem([self.acqXorigin, self.timebase_range])
            self.lr2.setZValue(-10)

            tstart = self.dSB_FFTTimeStart.value() * 1.0e-6
            tstop = self.dSB_FFTTimeStop.value() * 1.0e-6
            self.lr1.setRegion((tstart,tstop))
            self.lr2.setRegion((tstart,tstop))
 
            self.FIDWidget.addItem(self.lr1)
            self.FID2Widget.addItem(self.lr2)

        def acqInit(self):
            """
            """
            start = 0
            global probe
            global probe_avg

            if not self.hasConnectedDigitizer:
                self.digi = None
            elif not self.hasConnectedScope:
                self.scope = None
            else:
                print("Connect Digitizer or Scope")
                return

            probe = {}
            probe_avg = {}
            try:
                x = [self.acqXorigin + (start + i) * self.acqXfac for i in range(self.acqRecordlength)]
                for ch in self.measurement_settings.signal_channels:
                    probe[ch] = {}
                    probe_avg[ch] = {}
                    for i in range(self.measurement_settings.num_segments):
                        probe[ch][i] = spec.FID(x,
                                                np.zeros(self.acqRecordlength))
                        probe_avg[ch][i] = spec.FID(x,
                                                    np.zeros(self.acqRecordlength))

                self.updatePlots()

                self.acqWdir = self.lE_WorkDirectory.text()
                self.acqFilename = self.lE_Filename.text()
    #            if wdir is not None:
    #                if not os.path.exists(wdir):
    #                    os.makedirs(wdir)
    #                if wdir[-1] != '/':
    #                    wdir += '/'
                self.acqFilenameSettings = self.acqFilename + '.set'
    #            filename = wdir + filename

                # save settings
                self.save_settings(
                    self.acqFilenameSettings, 
                    comment = 'Jet Experiment: \n' )

                start = 0
                size = self.acqRecordlength

                self.sigAcqResetCounter.emit()
                self.sigAcqConfigure.emit(self.measurement_settings.totNumSegments, 
                                          self.measurement_settings.num_segments,
                                          self.acqFilename,
                                          self.awg,
                                          self.scope,
                                          self.digi,
                                          self.measurement_settings.marker_channel,
                                          self.measurement_settings.marker_point,
                                          self.measurement_settings.segmented_mode
                                      )

                self.acqThread.start()
            except Exception as e:
                print("Error during initialization: Correct Settings?")
                print(e)

        def acqContinue(self):
            """
            """
            if not self.hasConnectedDigitizer:
                self.digi = None
            elif not self.hasConnectedScope:
                self.scope = None
            else:
                print("Connect Digitizer or Scope")
                return

            # during acquisition the sum is stored and will always be devided by
            # the number of scans
            for channel in probe.keys():
                for j in range(len(probe_avg[channel])):
                    probe_avg[channel][j].y *= self.current_run

            
            
            self.sigAcqConfigure.emit(self.measurement_settings.totNumSegments, 
                                      self.measurement_settings.num_segments,
                                      self.acqFilename,
                                      self.awg,
                                      self.scope,
                                      self.digi,
                                      self.measurement_settings.marker_channel,
                                      self.measurement_settings.marker_point,
                                      self.measurement_settings.segmented_mode
                                    )

            self.acqThread.start()



        def acqSetStatus(self, status):
            self.lE_AcquisitionStatus.setText(status)

        def acqSetScanNumber(self, scan):
            self.current_run = scan
            self.lN_CurrentScan.display(scan)

# ----------------------------------
# Display
# ----------------------------------
        def lr1Update(self):
            start, stop = self.lr1.getRegion()
            print(start,stop)
            self.dSB_FFTTimeStart.setValue(start * 1.0e6)
            self.dSB_FFTTimeStop.setValue(stop * 1.0e6)
            self.lr2.setRegion((start,stop))
            #self.updateFFT()
#            self.fftThread.start()

        def lr2Update(self):
            start, stop = self.lr2.getRegion()
            print(start,stop)
            self.dSB_FFTTimeStart.setValue(start * 1.0e6)
            self.dSB_FFTTimeStop.setValue(stop * 1.0e6)
            self.lr1.setRegion((start,stop))
            #self.updateFFT()
 #           self.fftThread.start()

        def regionUpdate(self):
            tstart = self.dSB_FFTTimeStart.value() * 1.0e-6
            tstop = self.dSB_FFTTimeStop.value() * 1.0e-6
            self.lr1.setRegion((tstart, tstop))
            self.lr2.setRegion((tstart, tstop))

            #self.updateFFT()

        def FFTrun(self):

            try:
                tstart = self.dSB_FFTTimeStart.value() * 1.0e-6
                tstop = self.dSB_FFTTimeStop.value() * 1.0e-6
                item = self.coB_FFTSource.currentData()
                if self.hasConnectedScope:
                    samplerate = float(self.scope.settings.get('samplerate'))
                else:
                    samplerate = float(self.digi.settings.get('samplerate'))
                print(samplerate)

                unit = self.coB_FFTUnit.currentText()
                window_function = self.coB_WindowFunction.currentText()
                lo_freq = self.dSB_SynthFrequency.value() * 1.0e6 # in MHz

                segId = self.sB_SegmentId.value()

   #            ch1_avail = cB_AcquireChannel1.isChecked()
   #            ch3_avail = cB_AcquireChannel3.isChecked()

   #            showAVG = self.cB_DisplayAverage.isChecked()
                if 1 in probe_avg.keys():
                    i_sig_avail = True
                    chani = 1
                if 3 in probe_avg.keys():
                    q_sig_avail = True
                    chanq = 3
                if 2 in probe_avg.keys():
                    q_sig_avail = True
                    chanq = 2

                if i_sig_avail and q_sig_avail:
                    if item == 'AVG':
                        tsignal = probe_avg[1][segId].x
                        isignal = probe_avg[chani][segId].y / self.current_run
                        qsignal = probe_avg[chanq][segId].y / self.current_run
                    elif item == 'LS':
                        tsignal = probe[1][segId].x
                        isignal = probe[chani][segId].y
                        qsignal = probe[cahnq][segId].y

                    elif item == 'SAVG':
                        tsignal = probe[1][0].x
                        isignal = np.array([probe_avg[chani][seg].y for seg in \
                                            probe_avg[chani].keys()]).sum(axis=0) \
                                / (self.current_run * \
                                   self.measurement_settings.num_segments)
                        qsignal = np.array([probe_avg[chanq][seg].y for seg in \
                                       probe_avg[chanq].keys()]).sum(axis=0) \
                                / (self.current_run * \
                                   self.measurement_settings.num_segments)


                    isComplexFFT = True                   

                elif q_sig_avail:
                    if item == 'AVG':
                        tsignal = probe_avg[chanq][seg].x
                        isignal = probe_avg[chanq][seg].y/ self.current_run
                    elif item == 'LS':
                        tsignal = probe[chanq][seg].x
                        isignal = probe[chanq][seg].y
                    elif item == 'SAVG':
                        tsignal = probe[chanq][0].x
                        isignal = np.array([probe_avg[chanq][seg].y for seg in \
                                            probe_avg[chanq].keys()]).sum(axis=0) \
                                / (self.current_run * \
                                   self.measurement_settings.num_segments)

                    isComplexFFT = False

                elif i_sig_avail:
                    if item == 'AVG':
                        tsignal = probe_avg[chani][seg].x
                        isignal = probe_avg[chani][seg].y/ self.current_run
                    elif item == 'LS':
                        tsignal = probe[chani][seg].x
                        isignal = probe[chani][seg].y
                    elif item == 'SAVG':
                        tsignal = probe[chani][0].x
                        isignal = np.array([probe_avg[chani][seg].y for seg in \
                                            probe_avg[chani].keys()]).sum(axis=0) \
                                / (self.current_run * \
                                   self.measurement_settings.num_segments)

                    isComplexFFT = False

                else:
                    return
                
                print("Do FFT")
                if isComplexFFT:
                    self.sigControlParameter.emit(
                        tstart,
                        tstop,
                        unit,
                        isComplexFFT,
                        window_function,
                        samplerate
                    )
                    self.sigSetLoFreq.emit(
                        lo_freq)
                    self.sigSendData.emit(
                        tsignal,
                        isignal,
                        qsignal)

                else:
                    self.sigControlParameter.emit(
                        tstart,
                        tstop,
                        unit,
                        isComplexFFT,
                        window_function,
                        samplerate
                    )
                    self.sigSetLoFreq.emit(
                        lo_freq)
                    self.sigSendData.emit(
                        tsignal,
                        isignal,
                        None)

                self.fftThread.start()

            except Exception as e:
                print("FFT Init Error", e)

        @QtCore.pyqtSlot(object, object)
        def updateFFTWindow(self, x, y):
#            if self.current_run > 0:
#                y = y/self.current_run

            print("Update FFT Window")
            print(x[-1], y[-1])
            cfreq = self.dSB_CenterFrequency.value()
            span = self.dSB_FFTSpan.value() 
            minInt = self.dSB_FFTMinIntensity.value()
            maxInt = self.dSB_FFTMaxIntensity.value()
            lo_freq = self.dSB_SynthFrequency.value()  # in MHz
            
            self.FFTWidget.clear()
            self.FFTWidget.plot(x*1.0e-6 + lo_freq,y)
            self.FFTWidget.setXRange(cfreq - 0.5 * span, 
                                     cfreq + 0.5 * span,
                                     padding = 0)

#----------------------------------------------
# File I/O
#----------------------------------------------
        def save_settings(self, filename, comment = None):
            """
            prints the settings of all instruments to file

            """
            f = open(filename, 'w')

            # print AWG settings
            self.awg.read_settings()
            f.write("# ------------------------------------- \n")
            f.write("# AWG \n")
            f.write("# ------------------------------------- \n")

            for ck in sorted(self.awg.settings.__dict__.keys() ):
                f.write("%s: %s \n" % (ck, vars(self.awg.settings)[ck]))

            # print Scope settings
            if self.hasConnectedScope:
                self.scope.read_settings()
                f.write("# ------------------------------------- \n")
                f.write("# Scope \n")
                f.write("# ------------------------------------- \n")
                for ck in sorted(self.scope.settings.__dict__.keys() ):
                    f.write("%s: %s \n" % (ck, vars(self.scope.settings)[ck]))

            # print Digitizer settings
            if self.hasConnectedDigitizer:
              #  self.digi.read_settings()
                f.write("# ------------------------------------- \n")
                f.write("# Digitizer \n")
                f.write("# ------------------------------------- \n")
                for ck in sorted(self.digi.settings.__dict__.keys() ):
                    f.write("%s: %s \n" % (ck, vars(self.digi.settings)[ck]))


            # print Synthesizer settings
            f.write("# ------------------------------------- \n")
            f.write("# Synthesizer \n")
            f.write("# ------------------------------------- \n")

            #for ck in sorted(self.s1.settings.__dict__.keys() ):
            #    f.write("%s: %s \n" % (ck, vars(self.s1.settings)[ck]))

#            # print Delay Generator settings
#            f.write("# ------------------------------------- \n")
#            f.write("# Delay generator \n")
#            f.write("# ------------------------------------- \n")
#
#            for ck in sorted(self.dg.settings.__dict__.keys() ):
#                f.write("%s: %s \n" % (ck, vars(self.dg.settings)[ck]))

            # print General settings
            f.write("# ------------------------------------- \n")
            f.write("# General \n")
            f.write("# ------------------------------------- \n")

            f.write(" Comment: \n")
            if comment is not None:
                f.write(comment)
            # for ck in sorted(self.settings.__dict__.keys() ):
            #    f.write("%s: %s \n" % (ck, vars(self.settings)[ck]))

            f.close()
                               


# ----------------------------------------------
# Instruments
#------------------------------------------------


if __name__ == '__main__':
    # define GUI elements
    QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_X11InitThreads)
    qApp = QtGui.QApplication(sys.argv)
    mainGUI = ChirpGUI()

    # parse arguments
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-d", "--debug", action='store_true',
        help="whether to add extra print messages to the terminal")
    parser.add_argument("-c", "--config", help="a file from which to load settings")
    parser.add_argument("-p", "--plot", help="a file to load in the plot window")
    parser.add_argument("-p2", "--plot2", help="a file to load in the plot window (extra)")
    args = parser.parse_args()
    if args.debug:
        mainGUI.debugging = True
    if args.config and os.path.isfile(args.config):
        mainGUI.settingsLoad(filename=args.config)
    #if args.plot and os.path.isfile(args.plot):
    #        mainGUI.scanLoad(filename=args.plot)
    #if args.plot2 and os.path.isfile(args.plot2):
    #        mainGUI.scanLoad(filename=args.plot2, toSecond=True)
   
    # start GUI
    mainGUI.show()
    qApp.exec_()
    qApp.deleteLater()
    sys.exit()
